import ctypes as ct
import os
import platform
import struct
import sys

if sys.platform.startswith('win'):
    # The _winreg module has been renamed to winreg in Python 3.
    try:
        import _winreg as winreg
    except ImportError:
        import winreg as winreg


def load_dll(win_name=None, linux_name=None):
    """Load dll or shared object file as appropriate.

    On Windows:
        If environment variable 'KVDLLPATH' is set, will try and load DLL named
        win_name from that directory. Otherwise checks registry for CANlib SDK
        installation directory.

    On Linux:
        If environment variable 'KVDLLPATH' is set, will try and load shared
        object library named linux_name from that directory. Otherwise let
        the OS try and find a matching shared library object.
    """
    dir = os.getcwd()
    installDir = os.environ.get('KVDLLPATH')
    if installDir is None:
        if sys.platform.startswith('win'):
            if (platform.architecture()[0] == '32bit'):
                aKeyName = r"SOFTWARE\KVASER AB\CanlibSDK"
            else:
                aKeyName = r"SOFTWARE\Wow6432Node\KVASER AB\CanlibSDK"
            aReg = winreg.ConnectRegistry(None, winreg.HKEY_LOCAL_MACHINE)
            aKey = winreg.OpenKey(aReg, aKeyName)
            aValue = winreg.QueryValueEx(aKey, "")
            baseDir = str(aValue[0])

            if 8 * struct.calcsize("P") == 32:
                installDir = os.path.join(baseDir, "Bin")
            else:
                installDir = os.path.join(baseDir, "bin_x64")
            installDir = os.path.realpath(installDir)

    if installDir is not None:
        os.chdir(installDir)
        # Some DLL's are loaded manually so we add installDir to the PATH in
        # order to allow the OS to find them later when needed
        os.environ["PATH"] += os.pathsep + installDir

    # Load our dll and all dependencies
    try:
        # Windows supports all dll
        if sys.platform.startswith('win'):
            dllFile = win_name
            loadedDll = ct.WinDLL(dllFile)
        elif linux_name is not None:
            dllFile = linux_name
            loadedDll = ct.CDLL(dllFile)
    except Exception as e:
        print(e)
        print("Could be a missing dependancy dll for '%s'." % dllFile)
        print("(Directory for dll: '%s')\n" % installDir)
        os.chdir(dir)
        exit(1)
    os.chdir(dir)
    return loadedDll
