"""Wrapper module for the Kvaser converter library kvlclib.

This module wraps the Kvaser kvlclib dll. For more info, see the kvlclib
help files which are availible in the CANlib SDK.
http://www.kvaser.com/developer/canlib-sdk/
"""

import ctypes as ct
import inspect
import os

from . import dllLoader

kvlcOK = 0   # OK.
kvlcFail = -1   # Generic error.
kvlcERR_PARAM = -2   # Error in supplied parameters.
kvlcEOF = -3   # End of input file reached.
kvlcERR_NOT_IMPLEMENTED = -4   # Not implemented.
kvlcERR_FILE_ERROR = -5   # File I/O error.
kvlcERR_FILE_EXISTS = -6   # Output file already exists.
kvlcERR_INTERNAL_ERROR = -7   # Unhandled internal error.
kvlcERR_NULL_POINTER = -8   # Unexpected null pointer.
kvlcERR_FILE_TOO_LARGE = -10   # File size too large for specified format.
kvlcERR_TYPE_MISMATCH = -11   # Supplied parameter has incorrect type.
kvlcERR_NO_FREE_HANDLES = -12   # Too many open KvlcHandle handles.
# Missing call to kvlcSetInputFile or kvlcFeedSelectFormat.
kvlcERR_NO_INPUT_SELECTED = -13
kvlcERR_CONVERTING = -14   # Call failed since conversion is running.
kvlcERR_BUFFER_SIZE = -15   # Supplied buffer too small to hold the result.
kvlcERR_INVALID_LOG_EVENT = -30   # Event is unknown to converter.
kvlcERR_NO_TIME_REFERENCE = -31   # Required timestamp missing.
kvlcERR_TIME_DECREASING = -32   # Decreasing time between files.
kvlcERR_MIXED_ENDIANESS = -33   # Wrong data type in MDF.

FILE_FORMAT_KME24 = 1    # Input and output file format.
FILE_FORMAT_KME25 = 2    # Input and output file format.
FILE_FORMAT_VECTOR_ASC = 3    # Output file format.
FILE_FORMAT_CSV = 4    # Output file format.
FILE_FORMAT_PLAIN_ASC = 5    # Output file format.
FILE_FORMAT_MEMO_LOG = 6    # Input (internal device logfile format).
FILE_FORMAT_KME40 = 7    # Input and output file format.
FILE_FORMAT_VECTOR_BLF = 8    # Output file format.
FILE_FORMAT_KME50 = 9    # Input and output file format.
FILE_FORMAT_CSV_SIGNAL = 100    # Output file format.
FILE_FORMAT_MDF = 101    # Output file format.
FILE_FORMAT_MATLAB = 102    # Output file format.
FILE_FORMAT_J1587 = 103    # Output file format.
FILE_FORMAT_J1587_ALT = 104    # Obsolete.
FILE_FORMAT_FAMOS = 105    # Output file format.
FILE_FORMAT_MDF_SIGNAL = 106    # Output file format.
FILE_FORMAT_MDF_4X = 107    # Output file format.
FILE_FORMAT_MDF_4X_SIGNAL = 108    # Output file format.
FILE_FORMAT_XCP = 200    # Output file format.
FILE_FORMAT_FAMOS_XCP = 201    # Output file format.
FILE_FORMAT_DEBUG = 1000    # Reserved for debug.


PROPERTY_START_OF_MEASUREMENT = {
    'type': ct.c_int(), 'value': 1, 'name': "START_OF_MEASUREMENT"}
PROPERTY_FIRST_TRIGGER = {
    'type': ct.c_int(), 'value': 2, 'name': "FIRST_TRIGGER"}
PROPERTY_USE_OFFSET = {
    'type': ct.c_int(), 'value': 3, 'name': "USE_OFFSET"}
PROPERTY_OFFSET = {
    'type': ct.c_int64(), 'value': 4, 'name': "OFFSET"}
PROPERTY_CHANNEL_MASK = {
    'type': ct.c_uint(), 'value': 5, 'name': "CHANNEL_MASK"}
PROPERTY_HLP_J1939 = {
    'type': ct.c_int(), 'value': 6, 'name': "HLP_J1939"}
PROPERTY_CALENDAR_TIME_STAMPS = {
    'type': ct.c_int(), 'value': 7, 'name': "CALENDAR_TIME_STAMPS"}
PROPERTY_WRITE_HEADER = {
    'type': ct.c_int(), 'value': 8, 'name': "WRITE_HEADER"}
PROPERTY_SEPARATOR_CHAR = {
    'type': ct.c_char(), 'value': 9, 'name': "SEPARATOR_CHAR"}
PROPERTY_DECIMAL_CHAR = {
    'type': ct.c_char(), 'value': 10, 'name': "DECIMAL_CHAR"}
PROPERTY_ID_IN_HEX = {
    'type': ct.c_int(), 'value': 11, 'name': "ID_IN_HEX"}
PROPERTY_DATA_IN_HEX = {
    'type': ct.c_int(), 'value': 12, 'name': "DATA_IN_HEX"}
PROPERTY_NUMBER_OF_TIME_DECIMALS = {
    'type': ct.c_int(), 'value': 13, 'name': "NUMBER_OF_TIME_DECIMALS"}
PROPERTY_NAME_MANGLING = {
    'type': ct.c_int(), 'value': 14, 'name': "NAME_MANGLING"}
PROPERTY_FILL_BLANKS = {
    'type': ct.c_int(), 'value': 15, 'name': "FILL_BLANKS"}
PROPERTY_SHOW_UNITS = {
    'type': ct.c_int(), 'value': 16, 'name': "SHOW_UNITS"}
PROPERTY_ISO8601_DECIMALS = {
    'type': ct.c_int(), 'value': 17, 'name': "ISO8601_DECIMALS"}
PROPERTY_MERGE_LINES = {
    'type': ct.c_int(), 'value': 18, 'name': "MERGE_LINES"}
PROPERTY_RESAMPLE_COLUMN = {
    'type': ct.c_int(), 'value': 19, 'name': "RESAMPLE_COLUMN"}
PROPERTY_VERSION = {
    'type': ct.c_int(), 'value': 20, 'name': "VERSION"}
PROPERTY_SHOW_COUNTER = {
    'type': ct.c_int(), 'value': 21, 'name': "SHOW_COUNTER"}
PROPERTY_CROP_PRETRIGGER = {
    'type': ct.c_int(), 'value': 22, 'name': "CROP_PRETRIGGER"}
PROPERTY_ENUM_VALUES = {
    'type': ct.c_int(), 'value': 23, 'name': "ENUM_VALUES"}
PROPERTY_SIZE_LIMIT = {
    'type': ct.c_uint(), 'value': 24, 'name': "SIZE_LIMIT"}
PROPERTY_TIME_LIMIT = {
    'type': ct.c_uint(), 'value': 25, 'name': "TIME_LIMIT"}
PROPERTY_LIMIT_DATA_BYTES = {
    'type': ct.c_int(), 'value': 26, 'name': "LIMIT_DATA_BYTES"}
PROPERTY_CREATION_DATE = {
    'type': ct.c_int64(), 'value': 27, 'name': "CREATION_DATE"}
PROPERTY_OVERWRITE = {
    'type': ct.c_int(), 'value': 28, 'name': "OVERWRITE"}
PROPERTY_SIGNAL_BASED = {
    'type': None, 'value': 1001, 'name': "SIGNAL_BASED"}
PROPERTY_SHOW_SIGNAL_SELECT = {
    'type': None, 'value': 1002, 'name': "SHOW_SIGNAL_SELECT"}
PROPERTY_ATTACHMENTS = {
    'type': None, 'value': 1003, 'name': "ATTACHMENTS"}


class WriterFormat(object):
    """Helper class that encapsulates a Writer.

    You may list available Writers, and query properties.
    """

    @classmethod
    def getFirstWriterFormat(cls):
        """Get the first supported output format."""
        dll = Kvlclib.kvlcDll
        id_ = ct.c_int()
        dll.kvlcGetFirstWriterFormat(ct.byref(id_))
        return id_.value

    @classmethod
    def getNextWriterFormat(cls, previous_id):
        """Get the next supported output format."""
        dll = Kvlclib.kvlcDll
        id_ = ct.c_int()
        dll.kvlcGetNextWriterFormat(previous_id, ct.byref(id_))
        return id_.value

    def __init__(self, id_):
        self.dll = Kvlclib.kvlcDll
        self.id_ = id_
        self.name = "Unknown name"
        self.extension = "Unknown extension"
        self.description = "Unknown description"
        self.fname = inspect.currentframe().f_code.co_name

        text = ct.create_string_buffer(256)
        text_len = ct.c_int(ct.sizeof(text))
        self.dll.kvlcGetWriterName(self.id_, text, text_len)
        self.name = text.value.decode('utf-8')

        text_len = ct.c_int(ct.sizeof(text))
        self.dll.kvlcGetWriterExtension(self.id_, text, text_len)
        self.extension = text.value.decode('utf-8')

        text_len = ct.c_int(ct.sizeof(text))
        self.dll.kvlcGetWriterDescription(self.id_, text, text_len)
        self.description = text.value.decode('utf-8')

    def isPropertySupported(self, wr_property):
        """Check if specified write property is supported.

        Retuns True if the property is supported by output format.

        Args:
            wr_property (int): Any one of the defined PROPERTY_xxx
        """
        self.fname = inspect.currentframe().f_code.co_name
        supported = ct.c_int()
        # qqqmac check that error handling works in class WriterFormat
        self.dll.kvlcIsPropertySupported(self.id_, wr_property['value'],
                                         ct.byref(supported))
        return supported.value

    def getPropertyDefault(self, wr_property):
        """Get default value for property."""
        self.fname = inspect.currentframe().f_code.co_name
        if wr_property['type'] is None:
            buf = ct.c_bool()
        else:
            buf = wr_property['type']
        self.dll.kvlcGetWriterPropertyDefault(self.id_, wr_property['value'],
                                              ct.byref(buf), ct.sizeof(buf))
        return buf.value

    def __str__(self):
        text = "%4d: %s (.%s)" % (self.id_, self.name, self.extension)
        text += ", %s" % self.description
        return text


class KvlcError(Exception):
    """Base class for exceptions raised by the Kvlclib class.

    Looks up the error text in the kvlclib dll and presents it together with
    the error code and the wrapper function that triggered the exception.

    """

    def __init__(self, kvlclib, kvlcERR):
        super(KvlcError, self).__init__()
        self.kvlclib = kvlclib
        self.kvlc_err = kvlcERR

    def __kvlcGetErrorText(self):
        msg = ct.create_string_buffer(
            "Error description is missing.".encode('utf-8'))
        self.kvlclib.dll.kvlcGetErrorText(
            self.kvlc_err, msg, ct.sizeof(msg))
        return msg.value.decode()

    def __str__(self):
        print("KvlcError kvlc_err: ", self.kvlc_err)
        return "[KvlcError] %s: %s (%d)" % (self.kvlclib.fname,
                                            self.__kvlcGetErrorText(),
                                            self.kvlc_err)


class KvlcEndOfFile(KvlcError):
    """Exception used when EOF is reached on input file."""

    def __init__(self, kvlclib, kvlcERR):
        super(KvlcEndOfFile, self).__init__(kvlclib, kvlcERR)


class KvlcVersion(object):
    """Class that holds kvlclib version number."""

    def __init__(self, major, minor, build):
        self.major = major
        self.minor = minor
        self.build = build

    def __str__(self):
        """Present the version number as 'major.minor.build'."""
        return "%d.%d.%d" % (self.major, self.minor, self.build)


class Kvlclib(object):
    """Wrapper class for the Kvaser converter library kvlclib.

    This class wraps the Kvaser kvlclib dll. For more info, see the kvlclib
    help files which are availible in the CANlib SDK.
    http://www.kvaser.com/developer/canlib-sdk/
    """

    kvlcDll = dllLoader.load_dll(win_name='kvlclib.dll',
                                 linux_name=None)

    def __init__(self, filename, file_format):
        self.fname = inspect.currentframe().f_code.co_name
        self.dll = Kvlclib.kvlcDll
        self.handle = None
        self.format = file_format

        self.dll.kvlcGetErrorText.argtypes = [ct.c_int32,
                                              ct.c_char_p,
                                              ct.c_uint]
        self.dll.kvlcGetErrorText.errcheck = self._kvlc_err_check
        self.dll.kvlcGetVersion.argtypes = [ct.POINTER(ct.c_int),
                                            ct.POINTER(ct.c_int),
                                            ct.POINTER(ct.c_int)]
        self.dll.kvlcGetVersion.errcheck = self._kvlc_err_check
        self.dll.kvlcCreateConverter.argtypes = [ct.c_void_p,
                                                 ct.c_char_p,
                                                 ct.c_int]
        self.dll.kvlcCreateConverter.errcheck = self._kvlc_err_check
        self.dll.kvlcDeleteConverter.argtypes = [ct.c_void_p]
        self.dll.kvlcDeleteConverter.errcheck = self._kvlc_err_check
        self.dll.kvlcEventCount.argtypes = [ct.c_void_p,
                                            ct.POINTER(ct.c_uint)]
        self.dll.kvlcEventCount.errcheck = self._kvlc_err_check
        self.dll.kvlcConvertEvent.argtypes = [ct.c_void_p]
        self.dll.kvlcConvertEvent.errcheck = self._kvlc_err_check
        self.dll.kvlcSetInputFile.argtypes = [ct.c_void_p,
                                              ct.c_char_p,
                                              ct.c_int]
        self.dll.kvlcSetInputFile.errcheck = self._kvlc_err_check
        self.dll.kvlcNextInputFile.argtypes = [ct.c_void_p,
                                               ct.c_char_p]
        self.dll.kvlcNextInputFile.errcheck = self._kvlc_err_check

        # Reading from device is not supported yet:
        # kvlcFeedSelectFormat(KvLogCnvHandle handle, int format);
        # kvlcFeedLogEvent(KvLogCnvHandle handle, void *event);
        # kvlcFeedNextFile(KvLogCnvHandle handle);

        self.dll.kvlcIsOutputFilenameNew.argtypes = [ct.c_void_p,
                                                     ct.POINTER(ct.c_int)]
        self.dll.kvlcIsOutputFilenameNew.errcheck = self._kvlc_err_check
        self.dll.kvlcGetOutputFilename.argtypes = [ct.c_void_p,
                                                   ct.c_char_p,
                                                   ct.c_int]
        self.dll.kvlcGetOutputFilename.errcheck = self._kvlc_err_check
        self.dll.kvlcGetWriterPropertyDefault.argtypes = [ct.c_int,
                                                          ct.c_uint,
                                                          ct.c_void_p,
                                                          ct.c_size_t]
        self.dll.kvlcGetWriterPropertyDefault.errcheck = self._kvlc_err_check
        self.dll.kvlcSetProperty.argtypes = [ct.c_void_p,
                                             ct.c_uint,
                                             ct.c_void_p,
                                             ct.c_size_t]
        self.dll.kvlcSetProperty.errcheck = self._kvlc_err_check
        self.dll.kvlcGetProperty.argtypes = [ct.c_void_p,
                                             ct.c_uint,
                                             ct.c_void_p,
                                             ct.c_size_t]
        self.dll.kvlcGetProperty.errcheck = self._kvlc_err_check
        self.dll.kvlcIsOverrunActive.argtypes = [ct.c_void_p,
                                                 ct.POINTER(ct.c_int)]
        self.dll.kvlcIsOverrunActive.errcheck = self._kvlc_err_check
        self.dll.kvlcResetOverrunActive.argtypes = [ct.c_void_p]
        self.dll.kvlcResetOverrunActive.errcheck = self._kvlc_err_check
        self.dll.kvlcIsDataTruncated.argtypes = [ct.c_void_p,
                                                 ct.POINTER(ct.c_int)]
        self.dll.kvlcIsDataTruncated.errcheck = self._kvlc_err_check
        self.dll.kvlcResetDataTruncated.argtypes = [ct.c_void_p]
        self.dll.kvlcResetDataTruncated.errcheck = self._kvlc_err_check
        self.dll.kvlcAttachFile.argtypes = [ct.c_void_p,
                                            ct.c_char_p]
        self.dll.kvlcAttachFile.errcheck = self._kvlc_err_check
        self.dll.kvlcGetFirstWriterFormat.argtypes = [ct.POINTER(ct.c_int)]
        self.dll.kvlcGetFirstWriterFormat.errcheck = self._kvlc_err_check
        self.dll.kvlcGetNextWriterFormat.argtypes = [ct.c_int,
                                                     ct.POINTER(ct.c_int)]
        self.dll.kvlcGetNextWriterFormat.errcheck = self._kvlc_err_check
        self.dll.kvlcGetWriterName.argtypes = [ct.c_int,
                                               ct.c_char_p,
                                               ct.c_int]
        self.dll.kvlcGetWriterName.errcheck = self._kvlc_err_check
        self.dll.kvlcGetWriterExtension.argtypes = [ct.c_int,
                                                    ct.c_char_p,
                                                    ct.c_int]
        self.dll.kvlcGetWriterExtension.errcheck = self._kvlc_err_check
        self.dll.kvlcGetWriterDescription.argtypes = [ct.c_int,
                                                      ct.c_char_p,
                                                      ct.c_int]
        self.dll.kvlcGetWriterDescription.errcheck = self._kvlc_err_check
        self.dll.kvlcIsPropertySupported.argtypes = [ct.c_int, ct.c_uint,
                                                     ct.POINTER(ct.c_int)]
        self.dll.kvlcIsPropertySupported.errcheck = self._kvlc_err_check

        self.dll.kvlcAddDatabaseFile.argtypes = [ct.c_void_p, ct.c_char_p,
                                                 ct.c_uint]
        self.dll.kvlcAddDatabaseFile.errcheck = self._kvlc_err_check
        self.dll.kvlcAddDatabase.argtypes = [ct.c_void_p, ct.c_void_p,
                                             ct.c_uint]
        self.dll.kvlcAddDatabase.errcheck = self._kvlc_err_check

        self.handle = ct.c_void_p(None)
        self.dll.kvlcCreateConverter(ct.byref(self.handle),
                                     filename.encode(), file_format.id_)

    def _kvlc_err_check(self, result, func, arguments):
        if result == kvlcEOF:
            raise KvlcEndOfFile(self, kvlcEOF)
        if result < 0:
            raise KvlcError(self, result)
        return result

    def getVersion(self):
        """Get the kvlclib version number.

        Returns the kvlclib version number from the kvlclib DLL currently in
        use.

        Args:
            None

        Returns:
            version (kvmVersion): Major and minor version number

        """
        self.fname = inspect.currentframe().f_code.co_name
        major = ct.c_int()
        minor = ct.c_int()
        build = ct.c_int()
        self.dll.kvlcGetVersion(ct.byref(major), ct.byref(minor),
                                ct.byref(build))
        version = KvlcVersion(major.value, minor.value, build.value)
        return version

    def deleteConverter(self):
        """Delete the converter and close all files."""
        self.fname = inspect.currentframe().f_code.co_name
        if self.handle is not None:
            self.dll.kvlcDeleteConverter(self.handle)
            self.handle = None

    def attachFile(self, filename):
        """Attach file to be included in the output file.

        E.g. used to add a database or a movie to the output.

        Note that the output format must support the property
        PROPERTY_ATTACHMENTS.
        """
        self.fname = inspect.currentframe().f_code.co_name
        filename = os.path.realpath(filename)
        ct_filename = ct.c_char_p(filename.encode('utf-8'))
        self.dll.kvlcAttachFile(self.handle, ct_filename)

    def convertEvent(self):
        """Convert next event.

        Convert one event from input file and write it to output file.
        """
        self.fname = inspect.currentframe().f_code.co_name
        self.dll.kvlcConvertEvent(self.handle)

    def getOutputFilename(self):
        """Get the filename of the current output file."""
        self.fname = inspect.currentframe().f_code.co_name
        filename = ct.create_string_buffer(256)
        self.dll.kvlcGetOutputFilename(self.handle, filename,
                                       ct.sizeof(filename))
        return filename.value.decode('utf-8')

    def getPropertyDefault(self, wr_property):
        """Get default value for a writer property."""
        return self.format.getPropertyDefault(wr_property)

    def getProperty(self, wr_property):
        """Get current value for a writer property."""
        self.fname = inspect.currentframe().f_code.co_name
        buf = wr_property['type']
        self.dll.kvlcGetProperty(self.handle, wr_property['value'],
                                 ct.byref(buf), ct.sizeof(buf))
        return buf.value

    def isPropertySupported(self, wr_property):
        """Check if specified wr_property is supported by the current format.

        Retuns True if the property is supported by the current format.

        Args:
            wr_property (int): Any one of the defined PROPERTY_xxx
        """
        return self.format.isPropertySupported(wr_property)

    def nextInputFile(self, filename):
        """Select next input file."""
        self.fname = inspect.currentframe().f_code.co_name
        filename = os.path.realpath(filename)
        ct_filename = ct.c_char_p(filename.encode('utf-8'))
        self.dll.kvlcNextInputFile(self.handle, ct_filename)

    def eventCount(self):
        """Get extimated number of events left.

        Get the estimated number of remaining events in the input file. This
        can be useful for displaying progress during conversion.
        """
        self.fname = inspect.currentframe().f_code.co_name
        count = ct.c_uint(0)
        self.dll.kvlcEventCount(self.handle, ct.byref(count))
        return count.value

    def setProperty(self, wr_property, value):
        """Set a property value.

        Args:
            wr_property (int): Any one of the defined PROPERTY_xxx
        """
        self.fname = inspect.currentframe().f_code.co_name
        buf = wr_property['type']
        buf.value = value
        self.dll.kvlcSetProperty(self.handle, wr_property['value'],
                                 ct.byref(buf), ct.sizeof(buf))

    def IsOutputFilenameNew(self):
        """Check if the converter has created a new file.

        This is only true once after a a new file has been created. Used when
        splitting output into multiple files.
        """
        self.fname = inspect.currentframe().f_code.co_name
        updated = ct.c_int()
        self.dll.kvlcIsOutputFilenameNew(self.handle,
                                         ct.byref(updated))
        return updated.value

    def IsOverrunActive(self):
        """Get overrun status.

        Overruns can occur during logging with a Memorator if the bus load
        exceeds the logging capacity. This is very unusual, but can occur if a
        Memorator runs complex scripts and triggers.
        """
        self.fname = inspect.currentframe().f_code.co_name
        overrun = ct.c_int()
        self.dll.kvlcIsOverrunActive(self.handle, ct.byref(overrun))
        return overrun.value

    def resetOverrunActive(self):
        """Reset overrun status."""
        self.fname = inspect.currentframe().f_code.co_name
        self.dll.kvlcResetOverrunActive(self.handle)

    def IsDataTruncated(self):
        """Get truncation status.

        Truncation occurs when the selected output converter can't write all
        bytes in a data frame to file. This can happen if CAN FD data is
        extracted to a format that only supports up to 8 data bytes,
        e.g. FILE_FORMAT_KME40.

        Truncation can also happen if PROPERTY_LIMIT_DATA_BYTES is set
        to limit the number of data bytes in output.

        Returns True if data has been truncated
        """
        self.fname = inspect.currentframe().f_code.co_name
        truncated = ct.c_int()
        self.dll.kvlcIsDataTruncated(self.handle, ct.byref(truncated))
        return truncated.value

    def resetStatusTruncated(self):
        """Reset data trunctation status."""
        self.fname = inspect.currentframe().f_code.co_name
        self.dll.kvlcResetDataTruncated(self.handle)

    def setInputFile(self, filename, file_format):
        """Select input file.

        Args:
            filename (string): Name of input file
            file_format (int): Any of the supported input FILE_FORMAT_xxx
        """
        self.fname = inspect.currentframe().f_code.co_name
        filename = os.path.realpath(filename)
        ct_filename = ct.c_char_p(filename.encode('utf-8'))
        self.dll.kvlcSetInputFile(self.handle, ct_filename, file_format)
