import ctypes as ct

from ..exceptions import DllException
from .enums import Error


_all_errors_by_status = {}


def _remember(cls):
    _all_errors_by_status[cls.status] = cls
    return cls


def kvm_error(status):
    """Create and return an exception object corresponding to `status`"""
    if status in _all_errors_by_status:
        return _all_errors_by_status[status]()
    else:
        return KvmGeneralError(status)


class KvmError(DllException):
    @staticmethod
    def _get_error_text(status):
        try:
            from .wrapper import KvmLib
            msg = ct.create_string_buffer(80)
            KvmLib.dll.kvmGetErrorText(status, msg, ct.sizeof(msg))
            err_txt = msg.value.decode('utf-8')
        except:
            err_txt = "Unknown error text"
        err_txt += ' (%d)' % status
        return err_txt


class KvmGeneralError(KvmError):
    """A kvmlib error that does not (yet) have its own Exception

    WARNING: Do not explicitly catch this error, instead catch `KvmError`. Your
    error may at any point in the future get its own exception class, and so
    will no longer be of this type. The actual status code that raised any
    `KvmError` can always be accessed through a `status` attribute.

    """

    def __init__(self, status):
        self.status = status
        super(KvmGeneralError, self).__init__()


@_remember
class KvmDiskError(KvmError):
    status = Error.DISK_ERROR


@_remember
class KvmNoDisk(KvmDiskError):
    status = Error.NO_DISK


@_remember
class KvmDiskNotFormated(KvmDiskError):
    status = Error.NOT_FORMATTED


@_remember
class KvmNoLogMsg(KvmError):
    status = Error.NOLOGMSG
